<?php

namespace App\Http\Controllers\API;

use Illuminate\Http\Request;
use App\Http\Controllers\API\BaseController;
use App\Models\Route;
use App\Models\User;
use App\Models\VerificationCode;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
// use Validator;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;
use App\Services\SmsIrService;

class AuthController extends BaseController
{
    protected $smsService;

    public function __construct(SmsIrService $smsService)
    {
        $this->smsService = $smsService;
    }
    /**
     * Auth api
     *
     * @return \Illuminate\Http\Response
     */
    public function register(Request $request) //: JsonResponse
    {
        // return $request->all();
        $request->validate([
            'phone' => 'required|regex:/^00989\d{9}$/',
            'email' => 'nullable|email',
            'password' => 'required|min:6',
        ], [
            'phone.required' => 'شماره تلفن الزامی است.',
            'phone.regex' => 'شماره تلفن رو به صورت صحیح وارد نمایی. (09121234567)',
            'email.email' => 'ایمیل وارد شده معتبر نیست.',
            'password.required' => 'رمز عبور الزامی است.',
            'password.min' => 'کلمه عبور باید حداقل 8 کاراکتر داشته باشد.',
        ]);

        // if ($validator->fails()) {
        //     // دریافت تمام خطاها به صورت آرایه
        //     return response()->json([
        //         'errors' => $validator->errors()->toArray() // تمامی خطاها به صورت آرایه
        //     ], 422);
        // }

        $input = $request->all();
        $input['password'] = $input['password'] ?? bcrypt('#@$@#$sdfsdfsd$#%#$%2wef2344548');
        $input['code_confirm'] = rand(1000, 9999);

        $user = User::where('phone', $request->phone)->first();

        if (!$user) {
            $user = User::create($input);
        } else {
            $user->password = $input['password'];
            $user->code_confirm = $input['code_confirm'];
            $user->save();

            VerificationCode::create([
                'user_id' => $user->id,
                'type' => 'phone',
                'code' => $input['code_confirm'],
                'expires_at' => Carbon::now()->addMinutes(10)
            ]);
        }

        $this->smsService->sendVerifyCode(
            $request->phone, 
            $input['code_confirm'],
            true, // ارسال فوری
        );

        $success['token'] = $user->createToken('MyApp')->plainTextToken;
        $success['user'] =  $user;

        return $this->sendResponse($success, 'User register successfully.');
    }

    /**
     * Login api
     *
     * @return \Illuminate\Http\Response
     */
    public function login(Request $request) //: JsonResponse
    {
        $request->validate([
            'phone' => 'required|regex:/^00989\d{9}$/',
            'password' => 'required'
        ]);
        if (Auth::attempt(['phone' => $request->phone, 'password' => $request->password])) {
            $user = Auth::user();
            $success['token'] =  $user->createToken('MyApp')->plainTextToken;
            $success['user'] =  $user;

            return $this->sendResponse($success, 'User login successfully.');
        } else {

            return response()->json([
                'errors' => [
                    'phone' => 'شماره همراه وجود ندارد یا پسورد اشتباه است.'
                ],
            ], 422);
        }
    }

    public function sendVerificationCode(Request $request)
    {
        $request->validate([
            'phone' => 'required|regex:/^00989\d{9}$/',
            'email' => 'nullable|email' //|unique:users,email
        ], [
            // 'phone.nullable' => 'aaa',
            // 'phone.regex' => 'mmm',
        ]);

        // تولید کد تأیید 4 رقمی
        $user =  User::where('phone', $request->phone)->first();

        $code = rand(1000, 9999);

        if (!$user) {
            // return $this->sendError('حساب کاربری شما یافت نشد.', ['error' => 'حساب کاربری شما یافت نشد.']);
            $user = User::create([
                'phone' => $request->phone,
                'password' => bcrypt('#@$@#$sdfsdfsd$#%#$%2wef2344548'),
                'email' => ($request->email && $request->email != "") ? $request->email : $request->phone
                // 'code_confirm' => $code
            ]);
        }

        // ذخیره در دیتابیس
        VerificationCode::create([
            'user_id' => $user->id,
            'type' => 'phone',
            'code' => $code,
            'expires_at' => Carbon::now()->addMinutes(10)
        ]);

        $user->code_confirm = $code;
        $user->save();

        
        $this->smsService->sendVerifyCode(
            $request->phone, 
            $code,
            true, // ارسال فوری
        );

        // ارسال کد تأیید به ایمیل یا پیامک
        // Mail::to($user->email)->send(new SendVerificationCode($code));

        $success['user'] =  $user;

        return $this->sendResponse($success, 'کد تأیید ارسال شد.');

        // return response()->json(['message' => 'کد تأیید ارسال شد.', 'user' => $user], 200);
    }

    /**
     * Login api
     *
     * @return \Illuminate\Http\Response
     */
    public function verifyCode(Request $request)
    {
        $request->validate([
            'phone' => 'required|regex:/^00989\d{9}$/',
            'code' => 'required|numeric'
        ]);

        $user = User::where('phone', $request->phone)->where('code_confirm', $request->code)->first();

        if (!$user) {

            return response()->json([
                'errors' => ['code' => 'کد نامعتبر است.'],
            ], 422);
        }
        $VerificationCode = VerificationCode::where(['user_id' => $user->id, 'code' => $request->code])->first();

        if (!$VerificationCode) {
            return response()->json([
                'errors' => ['code' => 'کد نامعتبر است.'],
            ], 422);
        }

        if ($VerificationCode->isExpired()) {
            return response()->json([
                'errors' => ['code' => 'کد منقضی شده است.'],
            ], 422);
        }

        $newAccount = false;
        // حذف کد پس از تأیید
        $user->code_confirm = null;
        $user->phone_verified_at = Carbon::now();
        if (!$user->account_verified_at) {
            $newAccount = true;
            $user->account_verified_at = Carbon::now();
        }
        $user->save();

        VerificationCode::where(['user_id' => $user->id, 'code' => $request->code])->update([
            'code_verified_at' => Carbon::now()
        ]);

        $success['token'] = $user->createToken('MyApp')->plainTextToken;
        $success['user'] =  $user;

        $success['new_account'] = $newAccount;

        return $this->sendResponse($success, 'کد تأیید شد!');
    }

    public function getUser(Request $request)
    {
        $user = $request->user(); // دریافت کاربر احراز هویت شده
        // $user = User::with('accessibleRoutes', 'roles', 'permissions')->where('id', $u->id)->first();
        // $user->load('accessibleRoutes', 'roles', 'permissions');
        // return ($user);
        // $user->load('posts', 'comments'); // بارگذاری روابط
        $user->load('roles', 'permissions', 'messages', 'patients', 'country', 'state', 'city', 'treatments', 'time_slots'); // بارگذاری روابط
        $user->accessibleRoutes = $user->accessibleRoutes();
        $success['token'] = $user->createToken('MyApp')->plainTextToken;
        $success['user'] =  $user;

        // $success['accessibleRoutes'] = Route::whereHas('roles', function ($query, $user) {
        //     $query->whereIn('id', $user->roles->pluck('id'));
        // })
        //     ->orWhereHas('permissions', function ($query, $user) {
        //         $query->whereIn('id', $user->permissions->pluck('id'));
        //     })
        //     ->get();

        return $this->sendResponse($success, 'اطلاعات کاربر دریافت شد');
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'phone' => 'required|regex:/^00989\d{9}$/',
            'code' => 'required|numeric',
            'password' => 'required',
            'password_confirmation' => 'required|same:password',
        ]);

        $user = User::where('phone', $request->phone)->first();

        if (!$user) {
            return $this->sendError('کاربر نا معتبر است دوباره وارد حساب کاربری خود شوید.', ['error' => 'کاربر نا معتبر است دوباره وارد حساب کاربری خود شوید.']);
        }

        $VerificationCode = VerificationCode::where(['user_id' => $user->id, 'code' => $request->code])->first();

        if (!$VerificationCode) {
            return response()->json([
                'errors' => ['code' => 'کد نامعتبر است.'],
            ], 422);
        }

        if ($VerificationCode->isExpired()) {
            return response()->json([
                'errors' => ['code' => 'کد منقضی شده است.'],
            ], 422);
        }

        // حذف کد پس از تأیید
        $user->code_confirm = null;
        $user->password = bcrypt($request->password);
        $user->save();

        VerificationCode::where(['user_id' => $user->id, 'code' => $request->code])->update([
            'code_verified_at' => Carbon::now()
        ]);

        $success['token'] = $user->createToken('MyApp')->plainTextToken;
        $success['user'] =  $user;

        return $this->sendResponse($success, 'رمز عبور تنظیم گردید.');
    }

    
    public function updateUser(Request $request)
    {
        $request->validate([
            // 'user_id' => 'sometimes|exists:users,id',
            // 'uuid' => ['sometimes', Rule::unique('patients')->ignore($patient->id)],
            'name' => 'sometimes|string|max:255',
            'birth_day_at' => 'nullable|date',
            'father_name' => 'nullable|string|max:255',
            'mother_name' => 'nullable|string|max:255',
            'job_mother_or_father' => 'nullable|string|max:255',
            'reagent' => 'nullable|string|max:255',
            'country_id' => 'nullable|integer|exists:countries,id',
            'state_id' => 'nullable|integer|exists:states,id',
            'city_id' => 'nullable|integer|exists:cities,id',
            'address' => 'nullable|string',
            'confirmed_at' => 'nullable',
        ]);

        $user = Auth::guard('sanctum')->user();
        
        $user->national_code = $request->national_code ?? '';
        $user->name = $request->name;
        $user->birth_day_at = $request->birth_day_at ? Carbon::parse($request->birth_day_at) : null; // تبدیل تاریخ
        $user->father_name = $request->father_name;
        $user->mother_name = $request->mother_name;
        $user->job_mother_or_father = $request->job_mother_or_father;
        $user->reagent = $request->reagent;
        $user->sex = $request->sex;
        $user->marital_status = $request->marital_status;
        $user->home_phone = $request->home_phone;
        $user->work_phone = $request->work_phone;
        $user->country_id = $request->country_id === '' ? null : $request->country_id;
        $user->state_id = $request->state_id === '' ? null : $request->state_id;
        $user->city_id = $request->city_id === '' ? null : $request->city_id;
        $user->address = $request->address;
        // $user->confirmed_at = $request->confirmed_at ? Carbon::now() : null; // تبدیل تاریخ
        $user->save();

        return response()->json(['message' => 'اطلاعات بیمار به‌روزرسانی شد.', 'user' => $user]);
    }
}
