<?php

namespace App\Http\Controllers\API\V1;

use App\Http\Controllers\API\BaseController;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Patient;
use App\Models\Treatment;
use App\Models\TimeSlot;
use App\Models\Type;
use App\Models\Message;
use App\Models\AnswerQuestion;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class PatientsController extends BaseController
{
/**
     * نمایش لیست بیماران
     */
    public function index()
    {
        $page = request('page', 1);
        $perPage = min(request('per_page', 100), 200);
        
        $patients = Patient::query()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $page);
        
        return response()->json([
            'patients' => $patients->items(),
            'total' => $patients->total(),
            'current_page' => $patients->currentPage(),
            'per_page' => $patients->perPage(),
            'last_page' => $patients->lastPage()
        ]);
        // $success['patients'] = $patients = Patient::latest('uuid')
        // ->with('user','country','state','city')
        // ->get();

        // return $this->sendResponse($success, 'اطلاعات دریافت شد');
        // return response()->json($patients);
    }

    /**
     * ایجاد بیمار جدید
     */
    public function store(Request $request)
    {
        $request->validate([
            'phone' => 'required|regex:/^00989\d{9}$/',
            // 'user_id' => 'required|exists:users,id',
            // 'uuid' => 'required|unique:patients,uuid',
            'fullname' => 'required|string|max:255',
            // 'birth_day_at' => 'nullable|date',
            // 'father_name' => 'nullable|string|max:255',
            // 'mother_name' => 'nullable|string|max:255',
            // 'job_mother_or_father' => 'nullable|string|max:255',
            // 'reagent' => 'nullable|string|max:255',
            // 'country_id' => 'nullable|exists:countries,id',
            // 'state_id' => 'nullable|exists:states,id',
            // 'city_id' => 'nullable|exists:cities,id',
            // 'address' => 'nullable|string',
            // 'confirmed_at' => 'nullable|date',
        ]);
        // return $request;

        $user = User::where('phone', $request->phone)->first();
        if(!$user){
            $user = User::create([
                'phone' => $request->phone,
                'password' => bcrypt($request->phone),
                'name' => $request->fullname,
            ]);
        }
        $patient = Patient::where('user_id', $user->id)->where('fullname', $request->fullname)->first();
        
        $success['title'] = 'ذخیره اطلاعات';
        $success['message'] = 'ذخیره اطلاعات موفقیت آمیز بود.';
        if(!$patient){
            $patient = new Patient();
            $patient->user_id = $user->id;
            $patient->fullname = $request->fullname;
            $patient->save();
        }else{
            $success['title'] = 'ذخیره اطلاعات';
            $success['message'] = 'بیمار از قبل وجود داشت. لطفا از جستجو بیمار را پیدا نمایید.';
        }
        

        $success['patient'] = $patient;
        
        return $this->sendResponse($success, 'با موفقیت اضافه گردید.');
    }

    /**
     * نمایش جزئیات یک بیمار
     */
    public function show(Patient $patient)
    {
        // return response()->json($patient);
        $patient = Patient::where('id', $patient->id)->with('user', 'country', 'state', 'city', 'treatments', 'time_slots', 'answers')->first();
        
        
        // $success['questions'] = $types = Type::where('tags', 'like', "%سوالات-بیماریهای-قبلی%")
        //     ->with(['answer' => function($query)use($patient) {
        //         $query->where('user_id', $patient->user_id);
        //     }])
        //     ->get();
        
        // $success['types_visit'] = Type::where('tags', 'like', "%انواع-ویزیت%")->get();
        // $success['types_treatment'] = Type::where('tags', 'like', "%انواع-درمان%")->get();

        $success['types'] = $types = Type::where('tags', 'like', "%سوالات-بیماریهای-قبلی%")
            ->with(['answer' => function($query)use($patient) {
                $query->where('user_id', $patient->user_id);
            }])
            ->orWhere('tags', 'like', "%انواع-ویزیت%")
            ->orWhere('tags', 'like', "%انواع-درمان%")
            ->get();
            
        $success['types_visit'] = $types->filter(fn($type) => str_contains($type->tags, "انواع-ویزیت"))->values();
        $success['types_treatment'] = $types->filter(fn($type) => str_contains($type->tags, "انواع-درمان"))->values();
        $success['questions'] = $types->filter(fn($type) => str_contains($type->tags, "سوالات-بیماریهای-قبلی"))->values();


        $success['messages'] = Message::where('user_id', $patient->user_id)->with('user')->orderByDesc('created_at')->limit(10)->get();
        $success['time_slots'] = TimeSlot::where('user_id', $patient->user_id)->with('user','type')->latest('id')->get();

        
        $success['title'] = 'نمایش اطلاعات';
        $success['message'] = 'نمایش اطلاعات موفقیت آمیز بود.';
        $success['patient'] = $patient;
        
        return $this->sendResponse($success, 'OK');
    }

    public function answerQuestions(Request $request)
    {
        
        $request->validate([
            'id' => 'required|exists:patients,id',
            'questions' => 'nullable|array',
            'questions.*' => 'nullable|string|max:255', // برای همه عناصر آرایه
        ]);
        // return  $request;
        // return response()->json($patient);
        $success['patient'] = $patient = Patient::where('id', $request->id)->first();

        foreach ($request->questions as $key => $value) {
            # code...
            $answer = AnswerQuestion::updateOrCreate(
                [
                    'user_id' => $patient->user_id,
                    'patient_id' => $patient->id,
                    'type_id' => $key,
                ],
                [
                    'answer' => $value ? 'true' : 'false',
                    'description' => $value ?? '',
                ]
                );
        }
                
        
        $success['title'] = 'ذخیره اطلاعات';
        $success['message'] = 'ذخیره اطلاعات موفقیت آمیز بود.';
        return $this->sendResponse($success, $success['message']);
    }

    /**
     * به‌روزرسانی اطلاعات بیمار
     */
    public function update(Request $request, Patient $patient)
    {
        $request->validate([
            'user_id' => 'sometimes|exists:users,id',
            // 'uuid' => ['sometimes', Rule::unique('patients')->ignore($patient->id)],
            'fullname' => 'sometimes|string|max:255',
            'birth_day_at' => 'nullable|date',
            'father_name' => 'nullable|string|max:255',
            'mother_name' => 'nullable|string|max:255',
            'job_mother_or_father' => 'nullable|string|max:255',
            'reagent' => 'nullable|string|max:255',
            'country_id' => 'nullable|integer|exists:countries,id',
            'state_id' => 'nullable|integer|exists:states,id',
            'city_id' => 'nullable|integer|exists:cities,id',
            'address' => 'nullable|string',
            'confirmed_at' => 'nullable',
        ]);

        $patient->national_code = $request->national_code ?? '';
        $patient->fullname = $request->fullname;
        $patient->birth_day_at = $request->birth_day_at ? Carbon::parse($request->birth_day_at) : null; // تبدیل تاریخ
        $patient->father_name = $request->father_name;
        $patient->mother_name = $request->mother_name;
        $patient->job_mother_or_father = $request->job_mother_or_father;
        $patient->reagent = $request->reagent;
        $patient->sex = $request->sex;
        $patient->marital_status = $request->marital_status;
        $patient->home_phone = $request->home_phone;
        $patient->work_phone = $request->work_phone;
        $patient->country_id = $request->country_id === '' ? null : $request->country_id;
        $patient->state_id = $request->state_id === '' ? null : $request->state_id;
        $patient->city_id = $request->city_id === '' ? null : $request->city_id;
        $patient->address = $request->address;
        $patient->confirmed_at = $request->confirmed_at ? Carbon::now() : null; // تبدیل تاریخ
        $patient->save();

        return response()->json(['message' => 'اطلاعات بیمار به‌روزرسانی شد.', 'patient' => $patient]);
    }

    /**
     * حذف بیمار
     */
    public function destroy(Patient $patient)
    {
        $patient->delete();
        return response()->json(['message' => 'بیمار با موفقیت حذف شد.']);
    }

    public function getPatient($id){

        $user = Auth::guard('sanctum')->user();

        $patient = Patient::where([
                'id'=> $id,
                'user_id'=> $user->id
            ])
            ->with('user', 'country', 'state', 'city', 'treatments', 'time_slots', 'answers')->first();
                
        $success['title'] = 'نمایش اطلاعات';
        $success['message'] = 'نمایش اطلاعات موفقیت آمیز بود.';
        $success['patient'] = $patient;
        
        return $this->sendResponse($success, 'OK');
    }

}
