<?php

// namespace App\Http\Controllers;

namespace App\Http\Controllers\API\v1;

use App\Http\Controllers\API\BaseController;
use App\Services\TimeSlotGenerator;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Patient;
use App\Models\TimeSlot;
use App\Models\Type;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class TimeSlotController extends BaseController
{
    protected $generator;

    public function __construct(TimeSlotGenerator $generator)
    {
        $this->generator = $generator;
    }

    public function index(Request $request)
    {
        $query = TimeSlot::query();

        if ($request->has('from') && $request->has('to')) {
            $query->whereBetween('date_at', [$request->from, $request->to]);
        }
        
        // فقط نوبت‌های فعال نمایش داده شوند
        // $query->whereNotNull('user_id');
        
        // در صورت نیاز اطلاعات کاربری مرتبط هم به همراه هر نوبت فراخوانی شود
        $timeSlots = $query->with(['user', 'patient', 'type'])->get();

        return response()->json($timeSlots);
    }

    public function generate(Request $request)
    {
        // اعتبارسنجی توکن (اختیاری)
        // if ($request->token !== config('app.schedule_token')) {
        //     abort(403, 'Unauthorized');
        // }

        Log::info('Starting time slots generation via route');
        
        try {
            $count = $this->generator->generateForTwoWeeks();
            Log::info("Generated {$count} new time slots");
            return response()->json([
                'success' => true,
                'message' => "Generated {$count} new time slots"
            ]);
        } catch (\Exception $e) {
            Log::error('Generation failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    private function getPersianDayName($dayOfWeek)
    {
        $days = [
            0 => 'یکشنبه',
            1 => 'دوشنبه',
            2 => 'سه‌شنبه',
            3 => 'چهارشنبه',
            4 => 'پنج‌شنبه',
            5 => 'جمعه',
            6 => 'شنبه'
        ];
        
        return $days[$dayOfWeek] ?? 'نامعلوم';
    }

    public function timeSlotes(Request $request){
        $role = $request->role;
        $startDate = Carbon::today()->format('Y-m-d');
        $endDate = Carbon::today()->addMonth()->format('Y-m-d');
        
        $slots = TimeSlot::whereBetween('date_at', [$startDate, $endDate])
            // ->where('is_active', true)
            ->when($role === 'user', function ($query) {
                $query->whereTime('start_time', '>=', '18:00:00')
                      ->whereTime('start_time', '<', '19:00:00');
            })
            ->orderBy('date_at')
            ->orderBy('start_time')
            ->get()
            ->groupBy(function ($item) {
                return $item->date_at;
            })
            ->map(function ($daySlots, $date) {
                $carbonDate = Carbon::parse($date);
                
                return [
                    'date_at' => $date,
                    'day_name' => $this->getPersianDayName($carbonDate->dayOfWeek),
                    'slots' => $daySlots->map(function ($slot) {
                        return [
                            'id' => $slot->id,
                            'start_time' => Carbon::parse($slot->start_time)->format('H:i') ,
                            'end_time' => Carbon::parse($slot->end_time)->format('H:i'),
                            'user_id' => $slot->user_id,
                            'patient_id' => $slot->patient_id,
                            'is_active' => $slot->is_active ? true : false,
                        ];
                    })
                ];
            })
            ->values();


            $types = Type::where('tags', 'like', "%انواع-ویزیت%")->get();
        
        return response()->json([
            'success' => true,
            'slots' => $slots,
            'types' => $types,
        ]);
    }

    public function reserveSlot(Request $request){
        $request->validate([
            'slot_id' => 'required|exists:time_slots,id',
            'user_id' => 'nullable|exists:users,id',
            'patient_id' => 'nullable|exists:patients,id',
            'phone' => 'required|regex:/^00989\d{9}$/',
            'name' => 'required|string',
        ]);

        $user = User::where('phone', $request->phone)->first();
        if(!$user){
            $user = User::create([
                'phone' => $request->phone,
                'password' => bcrypt($request->phone),
                'name' => $request->name,
            ]);
        }
        $patient = Patient::where('id', $request->patient_id)->first();
        if(!$patient){
            $patient = Patient::where('user_id', $user->id)->first();
            if(!$patient){
                $patient = new Patient();
                $patient->user_id = $user->id;
                $patient->fullname = $request->name;
                $patient->save();
            }
        }
        
        $slot = TimeSlot::find($request->slot_id);
        
        if (!$slot->is_active) {
            return $this->sendError('این نوبت غیر فعال است', ['error' => 'این نوبت غیر فعال است'], 422);
        }
        if ($slot->user_id) {
            
            return $this->sendError('این نوبت قبلاً رزرو شده است', ['error' => 'این نوبت قبلاً رزرو شده است'], 422);
        }
        
        $slot->update([
            'user_id' => $user->id,
            'patient_id' => $patient->id,
            'type_id' => $request->type_id ?? null,
            'reservation_name' => $request->name,
            'reservation_phone' => $request->phone,
            // 'is_active' => false
        ]);

        $success['title'] = 'ذخیره اطلاعات';
        $success['message'] = 'ذخیره اطلاعات موفقیت آمیز بود.';
        return $this->sendResponse($success, $success['message']);
        
    }
}