<?php

namespace App\Http\Controllers\Api\v1;

use App\Http\Controllers\API\BaseController;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\TimeSlot;
use App\Models\User;
use App\Models\Treatment;
use App\Models\Type;
use App\Models\Patient;
use Carbon\Carbon;

class TreatmentController extends BaseController
{
    /**
     * Display a listing of the resource.
     */
    public function index(Patient $patient)
    {
        //
        
        $success['types_visit'] = Type::where('tags', 'like', "%انواع-ویزیت%")->get();
        $success['types_treatment'] = Type::where('tags', 'like', "%انواع-درمان%")->get();
        $success['treatments'] = Treatment::where('patient_id', $patient->id)->get();

        
        $success['title'] = 'ذخیره اطلاعات';
        $success['message'] = 'ذخیره اطلاعات موفقیت آمیز بود.';
        return $this->sendResponse($success, $success['message']);

    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
        $request->validate([
            'patient_id' => 'required|integer|exists:patients,id', // باید یک رشته JSON معتبر باشد
            'treatments' => 'nullable|array',
            'treatments.*' => 'nullable|numeric|min:1', // برای همه عناصر آرایه
            'time_slot_id' => 'nullable|exists:time_slots,id', // visit_id باید موجود در جدول visits باشد
            'visit_type' => 'nullable|exists:types,id', // اگر مقدار دارد باید یک رشته باشد
            'visit_date' => 'nullable|date_format:Y-m-d H:i', // فرمت تاریخ باید مطابق با yyyy-mm-dd hh:mm باشد
            'cost' => 'nullable|numeric|min:0', // هزینه باید عددی و بزرگتر از یا مساوی صفر باشد
            'payment_method' => 'nullable', // پرداخت باید یکی از این مقادیر باشد |in:insurance,cash,other
            'payment_description' => 'nullable|string', // توصیف پرداخت (اختیاری) باید رشته باشد
            'description' => 'nullable|string', // توصیف درمان (اختیاری) باید رشته باشد
            'treatment_at' => 'nullable|date_format:Y-m-d', // فرمت تاریخ باید مطابق با yyyy-mm-dd باشد
        ]);

        // return $request->treatments;;//->input('treatments[]');//, []);// $request->treatments;// json_decode($request->treatments);
        $patient = Patient::where('id', $request->patient_id)->first();

        $time_slot_id = $request->time_slot_id;
        if(!$request->time_slot_id){
            // $time_slot_id
            $visitDate = Carbon::parse($request->visit_date)->toDateString();
            $visitDateTime = Carbon::parse($request->visit_date)->format('H:i');

            $timeSlot = TimeSlot::where('start_time', $visitDateTime)
                ->where('date_at', $visitDate)->first();
            if($timeSlot){
                return response()->json([
                    'errors' => ['visit_date' => 'نوبت رزرو شده است.'],
                ], 422);
            }
            $request->validate([
                'visit_type' => 'required|exists:types,id', // اگر مقدار دارد باید یک رشته باشد
                'visit_date' => 'nullable|date_format:Y-m-d H:i',
            ]);
            $timeSlot = TimeSlot::create([
                'date_at' => $visitDate,
                'day_name' => $this->getDayOfWeekInPersian($request->visit_date),
                'start_time' => $visitDateTime,
                'end_time' => $visitDateTime,
                'is_active' => true,
                'user_id' => $patient->user->id,
                'patient_id' => $patient->id,
                'type_id' => $request->visit_type ?? null,
                'reservation_name' => $patient->user->name,
                'reservation_phone' => $patient->user->phone,
                // 'is_active' => false
            ]);
            $time_slot_id = $timeSlot->id;
        }
        $treatment = Treatment::find($request->id);
        if(!$treatment){
            $treatment = new Treatment();
        }
        $treatment->user_id = $patient->user_id;
        $treatment->patient_id = $patient->id;
        $treatment->time_slot_id = $time_slot_id;
        $treatment->cost = !empty($request->cost) ? $request->cost : 0;
        $treatment->payment_method = $request->payment_method;
        $treatment->payment_description = $request->payment_description;
        $treatment->description = $request->description;
        $treatment->treatment_at = $request->treatment_at ?? Carbon::now();
        $treatment->save();

        $treatment->types()->sync($request->treatments);

        
        $success['title'] = 'ذخیره اطلاعات';
        $success['message'] = 'ذخیره اطلاعات موفقیت آمیز بود.';
        $success['treatment'] = Treatment::where('id', $treatment->id)->with(['user', 'types', 'time_slot'])->latest()->first();
        $success['requests'] = $request->all();
        
        return $this->sendResponse($success, $success['message']);
    }

    function getDayOfWeekInPersian($timestamp)
    {
        // تبدیل timestamp به یک شیء Carbon
        $visitDate = Carbon::parse($timestamp);

        // گرفتن روز هفته به انگلیسی
        $dayOfWeek = $visitDate->format('l');

        // تبدیل روز هفته به فارسی
        $daysOfWeekInPersian = [
            'Sunday' => 'یک‌شنبه',
            'Monday' => 'دوشنبه',
            'Tuesday' => 'سه‌شنبه',
            'Wednesday' => 'چهارشنبه',
            'Thursday' => 'پنج‌شنبه',
            'Friday' => 'جمعه',
            'Saturday' => 'شنبه'
        ];

        // بازگشت معادل فارسی روز هفته
        return $daysOfWeekInPersian[$dayOfWeek] ?? null;
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
