<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Services\SmsIrService;

class SendQueuedSms implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $mobile;
    protected $message;
    protected $type;
    protected $parameters;
    protected $isBulk;

    /**
     * Create a new job instance.
     *
     * @param string|array $mobile شماره موبایل یا آرایه‌ای از شماره‌ها
     * @param string $message متن پیامک
     * @param string $type نوع پیامک (normal, verify, bulk)
     * @param array $parameters پارامترهای اضافی برای پیامک‌های verify
     * @param bool $isBulk آیا ارسال گروهی است؟
     */
    public function __construct($mobile, string $message = '', string $type = 'normal', array $parameters = [], bool $isBulk = false)
    {
        $this->mobile = $mobile;
        $this->message = $message;
        $this->type = $type;
        $this->parameters = $parameters;
        $this->isBulk = $isBulk;
    }

    /**
     * Execute the job.
     *
     * @param SmsIrService $smsService
     * @return void
     */
    public function handle(SmsIrService $smsService)
    {
        try {
            if ($this->type === 'verify') {
                // ارسال کد تأیید
                $smsService->sendImmediateVerify(
                    $this->mobile,
                    $this->parameters
                );
            } elseif ($this->isBulk) {
                // ارسال گروهی
                $smsService->sendImmediateBulkSms(
                    $this->mobile, // اینجا آرایه‌ای از شماره‌ها است
                    $this->message
                );
            } else {
                // ارسال عادی تک‌پیامک
                $smsService->sendImmediateSms(
                    $this->mobile,
                    $this->message
                );
            }
        } catch (\Exception $e) {
            // لاگ کردن خطا برای مشکل در پردازش Job
            \Log::error('خطا در پردازش Job ارسال پیامک', [
                'mobile' => is_array($this->mobile) ? implode(',', $this->mobile) : $this->mobile,
                'type' => $this->type,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // اگر خطا موقتی است، می‌توانیم Job را دوباره در صف قرار دهیم
            if ($this->attempts() < 3) {
                $this->release(30); // 30 ثانیه بعد دوباره امتحان کند
            }
        }
    }

    /**
     * تعیین تعداد دفعات تلاش برای اجرای Job
     */
    public function tries()
    {
        return 3;
    }

    /**
     * تعیین زمان انتظار بین تلاش‌های ناموفق
     */
    public function backoff()
    {
        return [30, 60, 120]; // 30 ثانیه، 1 دقیقه، 2 دقیقه
    }
}