<?php

namespace App\Services;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Arr;

class ModelStatsService
{
    /**
     * محاسبه آمار مدل
     *
     * @param string $modelClass مسیر کامل کلاس مدل مثلاً \App\Models\User
     * @param string $period "daily", "weekly", "monthly"
     */
    public function getStats(string $modelClass, string $period = 'monthly', array $conditions = []): array
    {
        if (!class_exists($modelClass) || !is_subclass_of($modelClass, Model::class)) {
            throw new \InvalidArgumentException('مدل نامعتبر است.');
        }
    
        $now = Carbon::now();
        $startOfCurrent = $this->getStartOfPeriod($now, $period);
        $startOfPrevious = $this->getStartOfPeriod($now->copy()->sub($this->periodToDateInterval($period)), $period);
        $endOfPrevious = $startOfCurrent->copy()->subSecond();
    
        // تابع اعمال شرایط روی کوئری
        $applyConditions = function ($query) use ($conditions) {
            foreach ($conditions as $condition) {
                if (!is_array($condition) || count($condition) < 2) continue;
    
                $column = $condition[0];
                $operator = strtolower($condition[1]);
    
                switch ($operator) {
                    case 'between':
                        $query->whereBetween($column, $condition[2] ?? []);
                        break;
                    case 'in':
                        $query->whereIn($column, $condition[2] ?? []);
                        break;
                    case 'not in':
                        $query->whereNotIn($column, $condition[2] ?? []);
                        break;
                    case 'like':
                    case 'not like':
                        $query->where($column, $operator, $condition[2] ?? '');
                        break;
                    default:
                        $query->where($column, $operator, $condition[2] ?? null);
                }
            }
            return $query;
        };
    
        // حالا کوئری‌ها با شرایط
        $total = $applyConditions($modelClass::query())->count();
    
        $col = class_basename($modelClass) == 'TimeSlot' ? 'date_at' : 'created_at';
        $current = $applyConditions($modelClass::query()->where($col, '>=', $startOfCurrent))->latest('id')->count();
    
        $previous = $applyConditions($modelClass::query()->whereBetween($col, [
            $startOfPrevious, $endOfPrevious
        ]))->count();
    
        $growth =  0;//$previous > 0 ? round((($current - $previous) / $previous) * 100, 2) : ($current > 0 ? 100 : 0);
    
        return [
            // previous: تعداد رکوردهایی که در بازه‌ی قبلی ثبت شده‌اند
            // مثلاً اگه period = 'monthly' باشه، این مقدار تعداد ثبت‌ها در ماه گذشته هست.
            // current: تعداد رکوردهایی که در بازه‌ی فعلی ثبت شده‌اند
            // مثلاً در همین ماه جاری، یا همین هفته، یا امروز — بسته به period.

            // 'growth_percentage' => $growth, // growth_percentage: درصد رشد نسبت به بازه‌ی قبل

            'model' => class_basename($modelClass),
            'total' => $total,
            'previous' => $previous,
            'current' => $current,
            'growth_percentage' => $growth,
            'period' => $period,
            'startOfCurrent' => $startOfCurrent,
            'startOfPrevious' => $startOfPrevious,
            'endOfPrevious' => $endOfPrevious,
        ];
    }
    

    private function getStartOfPeriod(Carbon $date, string $period): Carbon
    {
        return match ($period) {
            'daily' => $date->copy()->startOfDay(),
            'weekly' => $date->copy()->startOfWeek(),
            'monthly' => $date->copy()->startOfMonth(),
            default => throw new \InvalidArgumentException('بازه زمانی نامعتبر است.'),
        };
    }

    private function periodToDateInterval(string $period): \DateInterval
    {
        return match ($period) {
            'daily' => new \DateInterval('P1D'),
            'weekly' => new \DateInterval('P1W'),
            'monthly' => new \DateInterval('P1M'),
            default => throw new \InvalidArgumentException('بازه زمانی نامعتبر است.'),
        };
    }


    public function getModelWithRelations(string $modelClass, int $limit = 10, array $conditions = [])
    {
        if (!class_exists($modelClass) || !is_subclass_of($modelClass, Model::class)) {
            throw new \InvalidArgumentException('مدل نامعتبر است.');
        }

        $model = new $modelClass;

        // گرفتن تمام متدهای مدل
        $methods = get_class_methods($model);

        // فیلتر کردن متدهایی که یک رابطه Eloquent هستند
        $relations = method_exists($model, 'allowedRelations')
            ? $model->allowedRelations()
            : [];

        $query = $modelClass::with($relations);


        foreach ($conditions as $condition) {
            if (!is_array($condition) || count($condition) < 2) continue;
    
            $column = $condition[0];
            $operator = strtolower($condition[1]);
    
            switch ($operator) {
                case 'between':
                    $query->whereBetween($column, $condition[2] ?? []);
                    break;
    
                case 'in':
                    $query->whereIn($column, $condition[2] ?? []);
                    break;
    
                case 'not in':
                    $query->whereNotIn($column, $condition[2] ?? []);
                    break;
    
                case 'like':
                case 'not like':
                    $query->where($column, $operator, $condition[2] ?? '');
                    break;
    
                case 'or':
                    $query->orWhere($column, $condition[2] ?? '=' , $condition[3] ?? null);
                    break;
    
                default:
                    $query->where($column, $operator, $condition[2] ?? null);
            }
        }
        $colsort = class_basename($modelClass) == 'TimeSlot' ? 'date_at' : 'id';
        return $query
            ->latest($colsort)
            ->limit($limit)
            ->get();
    }
}


/*

🧪 استفاده در کنترلر

use App\Services\ModelStatsService;

public function stats(ModelStatsService $stats)
{
    return response()->json([
        'user_stats' => $stats->getStats(\App\Models\User::class, 'monthly'),
        'order_stats' => $stats->getStats(\App\Models\Order::class, 'weekly'),
    ]);
}
📤 خروجی JSON

{
  "user_stats": {
    "model": "User",
    "total": 1530,
    "previous": 120,
    "current": 180,
    "growth_percentage": 50,
    "period": "monthly"
  },
  "order_stats": {
    "model": "Order",
    "total": 3200,
    "previous": 90,
    "current": 130,
    "growth_percentage": 44.44,
    "period": "weekly"
  }
}


تابع بعدی
گرفتن رکوردهای اخر به همرا روابط هر مدل
use App\Services\ModelStatsService;

public function latestWithRelations(ModelStatsService $service)
{
    $data = $service->getModelWithRelations(\App\Models\User::class, 5);

    return response()->json($data);
}
    خروجی
    [
  {
    "id": 1,
    "name": "Ali",
    "email": "ali@example.com",
    "profile": { ... },
    "posts": [ ... ],
    "roles": [ ... ]
  },
  ...
]
*/