<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Queue;
use App\Jobs\SendQueuedSms;
use Carbon\Carbon;

class SmsIrService
{
    protected $apiKey;
    protected $lineNumber;
    protected $verifyTemplateId;

    public function __construct()
    {
        // $this->apiKey = config('services.smsir.api_key');
        // $this->lineNumber = config('services.smsir.line_number');
        // $this->verifyTemplateId = config('services.smsir.verify_template_id');
    }

    /**
     * ارسال پیامک با قابلیت انتخاب حالت فوری یا صف‌بندی شده
     * 
     * @param string $mobile
     * @param string $message
     * @param bool $immediate
     * @param array $extraParams
     * @param Carbon|null $sendAt زمان ارسال برای حالت زمانبندی شده
     * @return array
     */
    public function sendSms(string $mobile, string $message, bool $immediate = false, array $extraParams = [], ?Carbon $sendAt = null): array
    {
        if ($immediate || !empty($extraParams)) {
            return $this->sendImmediateSms($mobile, $message, $extraParams);
            return $this->sendImmediateSmsGet($mobile, $message, $extraParams);
        }
        
        return $this->queueSms($mobile, $message, $sendAt);
    }

    /**
     * ارسال گروهی پیامک
     * 
     * @param array $mobiles آرایه‌ای از شماره موبایل‌ها
     * @param string $message
     * @param bool $immediate
     * @param array $extraParams
     * @param Carbon|null $sendAt زمان ارسال برای حالت زمانبندی شده
     * @return array
     */
    public function sendBulkSms(array $mobiles, string $message, bool $immediate = false, array $extraParams = [], ?Carbon $sendAt = null): array
    {
        if ($immediate || !empty($extraParams)) {
            return $this->sendImmediateBulkSms($mobiles, $message, $extraParams);
        }
        
        return $this->queueBulkSms($mobiles, $message, $sendAt);
    }

    /**
     * ارسال کد تأیید با قابلیت انتخاب حالت فوری یا صف‌بندی شده
     * 
     * @param string $mobile
     * @param string $code
     * @param bool $immediate
     * @param array $extraParams
     * @param Carbon|null $sendAt زمان ارسال برای حالت زمانبندی شده
     * @return array
     */
    public function sendVerifyCode(string $mobile, string $code, bool $immediate = false, array $extraParams = [], ?Carbon $sendAt = null): array
    {
        $parameters = array_merge([
            ['name' => 'CODE', 'value' => $code]
        ], $extraParams);

        if ($immediate || !empty($extraParams)) {
            return $this->sendImmediateVerify($mobile, $parameters);
        }
        
        return $this->queueVerify($mobile, $parameters, $sendAt);
    }

    /**
     * ارسال فوری پیامک
     */
    protected function sendImmediateSms(string $mobile, string $message, array $extraParams = []): array
    {
        try {
            $payload = array_merge([
                'lineNumber' => $this->lineNumber,
                'mobile' => $mobile,
                'messageText' => $message,
            ], $extraParams);

            $response = Http::withHeaders([
                'X-API-KEY' => $this->apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ])->post('https://api.sms.ir/v1/send', $payload);

            $result = $response->json();

            if (!($result['status'] ?? false)) {
                $this->logError('خطا در ارسال فوری پیامک', $mobile, $result['message'] ?? 'Unknown error');
                return ['success' => false, 'message' => $result['message'] ?? 'Unknown error'];
            }
            
            if ($result['status'] !== 1) {
                Log::info('داده ناشناخته', [
                    'data' => json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE),
                    'mobile' => $mobile,
                    'key' => $this->apiKey,
                    'service' => 'SmsIrService',
                    'time' => now()->toDateTimeString()
                ]);
            }
            return ['success' => true, 'data' => $result];
        } catch (\Exception $e) {
            $this->logError('خطای غیرمنتظره در ارسال فوری پیامک', $mobile, $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    protected function sendImmediateSmsGet(string $mobile, string $message, array $extraParams = []): array
    {
        try {
            $payload = array_merge([
                'lineNumber' => $this->lineNumber,
                'mobile' => $mobile,//$mobile,
                'messageText' => $message,
            ], $extraParams);

            /*
            راهنمای جایگزینی پارامترها به شرح زیر است:
            username= نام کاربری (از منوی پروفایل > مشخصات حساب کاربری قابل مشاهده است)
            password= کلید وب سرویس (از منوی برنامه نویسان > لیست کلیدهای API قابل ایجاد است)
            line= شماره خط پیامکی (از منوی شماره پیامکی > شماره های من، قابل مشاهده است)
            mobile= شماره همراه گیرنده
            text= متن پیامک

            */

            $response = Http::withHeaders([
                'X-API-KEY' => $this->apiKey,
                'Accept' => 'application/json',
            ])->get('https://api.sms.ir/v1/send', [
                'username' => '9135368845',
                'password' => $this->apiKey,
                'line' => $this->lineNumber,
                'mobile' => $mobile,
                'text' => $message,
            ]);

            $result = $response->json();

            if (!($result['status'] ?? false)) {
                $this->logError('خطا در ارسال فوری پیامک', $mobile, $result['message'] ?? 'Unknown error');
                return ['success' => false, 'message' => $result['message'] ?? 'Unknown error'];
            }
            
            // if ($result['status'] !== 1) {
                Log::info('sms', [
                    'data' => json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE),
                    'mobile' => $mobile,
                    'key' => $this->apiKey,
                    'service' => 'SmsIrService',
                    
                    'username' => '9135368845',
                    'password' => $this->apiKey,
                    'line' => $this->lineNumber,
                    'mobile' => $mobile,
                    'text' => $message,
                    'X-API-KEY' => $this->apiKey,
                    'Accept' => 'application/json',
                    'time' => now()->toDateTimeString()
                ]);
            // }
            return ['success' => true, 'data' => $result];
        } catch (\Exception $e) {
            $this->logError('خطای غیرمنتظره در ارسال فوری پیامک', $mobile, $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * ارسال فوری گروهی پیامک
     */
    protected function sendImmediateBulkSms(array $mobiles, string $message, array $extraParams = []): array
    {
        try {
            $payload = array_merge([
                'lineNumber' => $this->lineNumber,
                'messageText' => $message,
                'mobiles' => $mobiles,
            ], $extraParams);

            $response = Http::withHeaders([
                'X-API-KEY' => $this->apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ])->post('https://api.sms.ir/v1/send/bulk', $payload);

            $result = $response->json();

            if (!($result['status'] ?? false)) {
                $this->logError('خطا در ارسال فوری گروهی پیامک', implode(',', $mobiles), $result['message'] ?? 'Unknown error');
                return ['success' => false, 'message' => $result['message'] ?? 'Unknown error'];
            }
            
            if ($result['status'] !== 1) {
                Log::info('داده ناشناخته', [
                    'data' => json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE),
                    'mobile' => $mobiles,
                    'key' => $this->apiKey,
                    'service' => 'SmsIrService',
                    'time' => now()->toDateTimeString()
                ]);
            }

            return ['success' => true, 'data' => $result];
        } catch (\Exception $e) {
            $this->logError('خطای غیرمنتظره در ارسال فوری گروهی پیامک', implode(',', $mobiles), $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * ارسال فوری کد تأیید
     */
    protected function sendImmediateVerify(string $mobile, array $parameters): array
    {
        try {
            $response = Http::withHeaders([
                'X-API-KEY' => $this->apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ])->post('https://api.sms.ir/v1/send/verify', [
                'mobile' => $mobile,
                'templateId' => $this->verifyTemplateId,
                'parameters' => $parameters
            ]);

            $result = $response->json();

            if (!($result['status'] ?? false)) {
                $this->logError('خطا در ارسال فوری کد تأیید', $mobile, $result['message'] ?? 'Unknown error');
                return ['success' => false, 'message' => $result['message'] ?? 'Unknown error'];
            }
            
            if ($result['status'] !== 1) {
                Log::info('داده ناشناخته', [
                    'data' => json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE),
                    'mobile' => $mobile,
                    'key' => $this->apiKey,
                    'service' => 'SmsIrService',
                    'time' => now()->toDateTimeString()
                ]);
            }
            
            return ['success' => true, 'data' => $result];
        } catch (\Exception $e) {
            $this->logError('خطای غیرمنتظره در ارسال فوری کد تأیید', $mobile, $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * قرار دادن پیامک در صف
     */
    protected function queueSms(string $mobile, string $message, ?Carbon $sendAt = null): array
    {
        try {
            $job = new SendQueuedSms($mobile, $message, 'normal');
            
            if ($sendAt) {
                Queue::later($sendAt, $job);
            } else {
                Queue::push($job);
            }
            
            return ['success' => true, 'message' => 'پیامک در صف ارسال قرار گرفت'];
        } catch (\Exception $e) {
            $this->logError('خطا در قرار دادن پیامک در صف', $mobile, $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * قرار دادن پیامک گروهی در صف
     */
    protected function queueBulkSms(array $mobiles, string $message, ?Carbon $sendAt = null): array
    {
        try {
            $job = new SendQueuedSms($mobiles, $message, 'bulk');
            
            if ($sendAt) {
                Queue::later($sendAt, $job);
            } else {
                Queue::push($job);
            }
            
            return ['success' => true, 'message' => 'پیامک گروهی در صف ارسال قرار گرفت'];
        } catch (\Exception $e) {
            $this->logError('خطا در قرار دادن پیامک گروهی در صف', implode(',', $mobiles), $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * قرار دادن کد تأیید در صف
     */
    protected function queueVerify(string $mobile, array $parameters, ?Carbon $sendAt = null): array
    {
        try {
            $job = new SendQueuedSms($mobile, '', 'verify', $parameters);
            
            if ($sendAt) {
                Queue::later($sendAt, $job);
            } else {
                Queue::push($job);
            }
            
            return ['success' => true, 'message' => 'کد تأیید در صف ارسال قرار گرفت'];
        } catch (\Exception $e) {
            $this->logError('خطا در قرار دادن کد تأیید در صف', $mobile, $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * لاگ کردن خطاها
     */
    protected function logError(string $title, string $mobile, string $error): void
    {
        Log::error($title, [
            'mobile' => $mobile,
            'error' => $error,
            'service' => 'SmsIrService',
            'time' => now()->toDateTimeString()
        ]);
    }

    function normalizePhoneNumbers(array $phones): array
    {
        return array_map(function($phone) {
            // حذف همه کاراکترهای غیرعددی
            $phone = preg_replace('/[^0-9]/', '', $phone);
            
            // حذف پیش‌وندهای اضافی
            if (str_starts_with($phone, '0098')) {
                return '0' . substr($phone, 4);
            }
            
            if (str_starts_with($phone, '98')) {
                return '0' . substr($phone, 2);
            }
            
            if (str_starts_with($phone, '+98')) {
                return '0' . substr($phone, 3);
            }
            
            return $phone;
        }, $phones);
    }

    public function sendLikeToLike(array $mobiles, array $messageTexts, ?int $sendDateTime = null): array
    {
        // اعتبارسنجی تعداد مساوی شماره‌ها و پیام‌ها
        if (count($mobiles) !== count($messageTexts)) {
            throw new \InvalidArgumentException('تعداد شماره موبایل‌ها و متن پیامک‌ها باید برابر باشد');
        }

        // اعتبارسنجی حداکثر 100 شماره
        if (count($mobiles) > 100) {
            throw new \InvalidArgumentException('حداکثر تعداد مجاز شماره‌های مقصد 100 می‌باشد');
        }

        // اعتبارسنجی زمان ارسال (اگر وجود دارد)
        if ($sendDateTime && $sendDateTime < (time() + 3600)) {
            throw new \InvalidArgumentException('زمان ارسال باید حداقل یک ساعت آینده باشد');
        }

        try {
            $payload = [
                'lineNumber' => $this->lineNumber,
                'messageTexts' => $messageTexts,
                'mobiles' => $mobiles
            ];

            if ($sendDateTime) {
                $payload['sendDateTime'] = $sendDateTime;
            }

            $response = Http::withHeaders([
                'x-api-key' => $this->apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json'
            ])->post('https://api.sms.ir/v1/send/likeToLike', $payload);

            $result = $response->json();

            if ($response->successful()) {
                Log::info('پیامک‌های نظیر به نظیر با موفقیت ارسال شدند', [
                    'pack_id' => $result['packId'] ?? null,
                    'message_ids' => $result['messageIds'] ?? [],
                    'cost' => $result['cost'] ?? 0,
                    'count' => count($mobiles)
                ]);
                
                return [
                    'success' => true,
                    'data' => $result,
                    'message' => 'پیامک‌ها با موفقیت ارسال شدند'
                ];
            } else {
                Log::error('خطا در ارسال پیامک‌های نظیر به نظیر', [
                    'error' => $result['message'] ?? 'خطای ناشناخته',
                    'status' => $response->status()
                ]);
                
                return [
                    'success' => false,
                    'error' => $result['message'] ?? 'خطای ناشناخته از سرویس SMS.ir',
                    'status' => $response->status()
                ];
            }
        } catch (\Exception $e) {
            Log::error('خطا در ارتباط با سرویس SMS.ir', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
        return [];
    }
}


/*
1. ارسال کد تأیید (Verify) بدون زمان‌بندی
use App\Services\SmsIrService;

$smsService = new SmsIrService();
$result = $smsService->sendVerifyCode(
    mobile: '09123456789', 
    code: '12345',
    immediate: true // این پارامتر باعث ارسال فوری می‌شود
);

if ($result['success']) {
    // پیامک با موفقیت ارسال شد
} else {
    // خطا در ارسال پیامک
    echo $result['message'];
}


2. ارسال پیامک عادی بدون زمان‌بندی
use App\Services\SmsIrService;

$smsService = new SmsIrService();
$result = $smsService->sendSms(
    mobile: '09123456789',
    message: 'متن پیامک شما',
    immediate: true // ارسال فوری
);

if ($result['success']) {
    // پیامک با موفقیت ارسال شد
} else {
    // خطا در ارسال پیامک
    echo $result['message'];
}


3. ارسال گروهی بدون زمان‌بندی
use App\Services\SmsIrService;

$smsService = new SmsIrService();
$result = $smsService->sendBulkSms(
    mobiles: ['09123456789', '09129876543'],
    message: 'متن پیامک گروهی',
    immediate: true // ارسال فوری
);

if ($result['success']) {
    // پیامک‌ها با موفقیت ارسال شدند
} else {
    // خطا در ارسال پیامک‌ها
    echo $result['message'];
}
    *****************************************
    /////////////////////////////////////////

ارسال تکی عادی:
Queue::push(new SendQueuedSms('09123456789', 'متن پیامک'));

// یا زمانبندی شده
Queue::later($when, new SendQueuedSms('09123456789', 'متن پیامک'));

ارسال گروهی:
Queue::push(new SendQueuedSms(
    ['09123456789', '09129876543'], 
    'متن پیامک گروهی',
    'normal',
    [],
    true // isBulk
));

// یا زمانبندی شده
Queue::later($when, new SendQueuedSms(
    ['09123456789', '09129876543'], 
    'متن پیامک گروهی',
    'normal',
    [],
    true
));

ارسال کد تأیید:
Queue::push(new SendQueuedSms(
    '09123456789',
    '',
    'verify',
    [['name' => 'CODE', 'value' => '12345']]
));

// یا زمانبندی شده
Queue::later($when, new SendQueuedSms(
    '09123456789',
    '',
    'verify',
    [['name' => 'CODE', 'value' => '12345']]
));
///////////////////////////
نحوه استفاده:

ارسال عادی با زمانبندی:
$smsService->sendSms(
    '09123456789', 
    'متن پیامک', 
    false, 
    [], 
    Carbon::now()->addHours(2) // ارسال 2 ساعت بعد
);

ارسال گروهی فوری:
$smsService->sendBulkSms(
    ['09123456789', '09129876543'],
    'متن پیامک گروهی',
    true
);

ارسال گروهی زمانبندی شده:
$smsService->sendBulkSms(
    ['09123456789', '09129876543'],
    'متن پیامک گروهی',
    false,
    [],
    Carbon::tomorrow()->setHour(9) // ارسال فردا ساعت 9 صبح
);


use App\Services\SmsIrService;

class AuthController extends Controller
{
    protected $smsService;

    public function __construct(SmsIrService $smsService)
    {
        $this->smsService = $smsService;
    }

    // ارسال فوری با پارامتر اضافه
    public function sendImmediateWithExtra()
    {
        $mobile = '09123456789';
        $code = rand(1000, 9999);
        
        $result = $this->smsService->sendVerifyCode(
            $mobile, 
            $code,
            true, // ارسال فوری
            [
                ['name' => 'NAME', 'value' => 'محمد']
            ]
        );

        return response()->json($result);
    }

    // ارسال معمولی (صف‌بندی شده)
    public function sendNormal()
    {
        $mobile = '09123456789';
        $message = 'این یک پیام تست است';
        
        $result = $this->smsService->sendSms($mobile, $message);

        return response()->json($result);
    }

    // ارسال فوری
    public function sendImmediate()
    {
        $mobile = '09123456789';
        $message = 'این یک پیام فوری است';
        
        $result = $this->smsService->sendSms($mobile, $message, true);

        return response()->json($result);
    }
}
*/