<?php

namespace App\Services;

use App\Models\TimeSlot;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class TimeSlotGenerator
{
    public function generateForTwoWeeks()
    {
        $startDate = now();
        $endDate = now()->addWeeks(2)->endOfWeek(); // دو هفته آینده
        
        Log::info("Generating time slots from {$startDate->toDateString()} to {$endDate->toDateString()}");
        
        $generatedCount = $this->generateSlotsBetweenDates($startDate, $endDate);
        
        Log::info("Generated {$generatedCount} new time slots");
        
        return $generatedCount;
    }

    public function generateSlotsBetweenDates(Carbon $startDate, Carbon $endDate): int
    {
        $currentDate = $startDate->copy();
        $count = 0;

        while ($currentDate <= $endDate) {
            if (!$currentDate->isFriday()) {
                $count += $this->createSlotsForDate($currentDate);
            }
            $currentDate->addDay();
        }

        return $count;
    }

    private function createSlotsForDate(Carbon $date): int
    {
        $startTime = $date->copy()->setTime(18, 0, 0);
        $endTime = $date->copy()->setTime(19, 0, 0);
        $currentSlot = $startTime->copy();
        $createdCount = 0;

        while ($currentSlot < $endTime) {
            $slotEnd = $currentSlot->copy()->addMinutes(5);

            // بررسی دقیق وجود نداشتن اسلات تکراری
            $exists = TimeSlot::whereDate('date_at', $date->format('Y-m-d'))
                ->where('start_time', $currentSlot->format('H:i:s'))
                ->where('end_time', $slotEnd->format('H:i:s'))
                ->exists();

            if (!$exists) {
                TimeSlot::create([
                    'date_at' => $date->format('Y-m-d'),
                    'day_name' => $this->getPersianDayName($date->dayOfWeek),
                    'start_time' => $currentSlot->format('H:i:s'),
                    'end_time' => $slotEnd->format('H:i:s'),
                    'is_active' => true
                ]);
                $createdCount++;
            }

            $currentSlot->addMinutes(5);
        }

        return $createdCount;
    }

    private function getPersianDayName(int $dayOfWeek): string
    {
        $days = [
            0 => 'یکشنبه',
            1 => 'دوشنبه',
            2 => 'سه‌شنبه',
            3 => 'چهارشنبه',
            4 => 'پنجشنبه',
            5 => 'جمعه',
            6 => 'شنبه'
        ];

        return $days[$dayOfWeek] ?? 'نامعلوم';
    }
}